/*

 PostgreSQL Schema

 Source Database       : broadstreet
 Source Schema         : public

 This file is used to create CARD database structure 

*/

/*
|--------------------------------------------------------------------------
| CREATE cv TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cv (
	cv_id serial PRIMARY KEY,
	name varchar(255) NOT NULL,
	definition text
);

-- --------------------------------------------------------------------------
--  Comments for table cv
-- --------------------------------------------------------------------------

COMMENT ON TABLE cv IS 'A controlled vocabulary or ontology. A cv is
composed of cvterms (AKA terms, classes, types, universals - relations
and properties are also stored in cvterm) and the relationships
between them.';

COMMENT ON COLUMN cv.name IS 'The name of the ontology. This
corresponds to the obo-format -namespace-. cv names uniquely identify
the cv. In OBO file format, the cv.name is known as the namespace.';

COMMENT ON COLUMN cv.definition IS 'A text description of the criteria for
membership of this ontology.';

/*
|--------------------------------------------------------------------------
| CREATE db TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE db (
	db_id serial PRIMARY KEY,
	name varchar(255) NOT NULL,
	description varchar(255),
	urlprefix varchar(255),
	url varchar(255)
);

-- --------------------------------------------------------------------------
--  Comments for table db
-- --------------------------------------------------------------------------

COMMENT ON TABLE db IS 'A database authority. Typical databases in
bioinformatics are FlyBase, GO, UniProt, NCBI, MGI, etc. The authority
is generally known by this shortened form, which is unique within the
bioinformatics and biomedical realm.  To Do - add support for URIs,
URNs (e.g. LSIDs). We can do this by treating the URL as a URI -
however, some applications may expect this to be resolvable - to be
decided.';


/*
|--------------------------------------------------------------------------
| CREATE dbxref TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE dbxref (
	dbxref_id serial PRIMARY KEY,
	db_id int4 NOT NULL REFERENCES db, 
	accession varchar(255) NOT NULL,
	version varchar(255) NOT NULL,
	description text,
	accession_tsvector tsvector
);

-- --------------------------------------------------------------------------
--  Comments for table dbxref
-- --------------------------------------------------------------------------

COMMENT ON TABLE dbxref IS 'A unique, global, public, stable identifier. Not necessarily an external reference - can reference data items inside the particular chado instance being used. Typically a row in a table can be uniquely identified with a primary identifier (called dbxref_id); a table may also have secondary identifiers (in a linking table <T>_dbxref). A dbxref is generally written as <DB>:<ACCESSION> or as <DB>:<ACCESSION>:<VERSION>.';
COMMENT ON COLUMN dbxref.accession IS 'The local part of the identifier. Guaranteed by the db authority to be unique for that db.';

/*
|--------------------------------------------------------------------------
| CREATE cvterm TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvterm (
	cvterm_id serial PRIMARY KEY,
	cv_id int4 REFERENCES cv,
	name varchar(1024) NOT NULL,
	definition text,
	dbxref_id int4 REFERENCES dbxref,
	is_obsolete int4 NOT NULL DEFAULT 0,
	is_relationshiptype int4 NOT NULL DEFAULT 0,
	name_tsvector tsvector,
	definition_tsvector tsvector
);


-- --------------------------------------------------------------------------
--  Comments for table cvterm
-- --------------------------------------------------------------------------

COMMENT ON TABLE cvterm IS 'A term, class, universal or type within an
ontology or controlled vocabulary.  This table is also used for
relations and properties. cvterms constitute nodes in the graph
defined by the collection of cvterms and cvterm_relationships.';

COMMENT ON COLUMN cvterm.cv_id IS 'The cv or ontology or namespace to which
this cvterm belongs.';

COMMENT ON COLUMN cvterm.name IS 'A concise human-readable name or
label for the cvterm. Uniquely identifies a cvterm within a cv.';

COMMENT ON COLUMN cvterm.definition IS 'A human-readable text
definition.';

COMMENT ON COLUMN cvterm.dbxref_id IS 'Primary identifier dbxref - The
unique global OBO identifier for this cvterm.  Note that a cvterm may
have multiple secondary dbxrefs - see also table: cvterm_dbxref.';

COMMENT ON COLUMN cvterm.is_obsolete IS 'Boolean 0=false,1=true; see
GO documentation for details of obsoletion. Note that two terms with
different primary dbxrefs may exist if one is obsolete.';

COMMENT ON COLUMN cvterm.is_relationshiptype IS 'Boolean
0=false,1=true relations or relationship types (also known as Typedefs
in OBO format, or as properties or slots) form a cv/ontology in
themselves. We use this flag to indicate whether this cvterm is an
actual term/class/universal or a relation. Relations may be drawn from
the OBO Relations ontology, but are not exclusively drawn from there.';


/*
|--------------------------------------------------------------------------
| CREATE cvterm_relationship TABLE
|--------------------------------------------------------------------------
|
| The cvterm_relationship statement is about the subject, not the object. For example "insect wing part_of thorax".
|
*/

CREATE TABLE cvterm_relationship (
	cvterm_relationship_id serial PRIMARY KEY,
	type_id int4 NOT NULL REFERENCES cvterm,
	subject_id int4 NOT NULL REFERENCES cvterm,
	object_id int4 NOT NULL REFERENCES cvterm
);

-- --------------------------------------------------------------------------
--  Comments for table cvterm_relationship
-- --------------------------------------------------------------------------

COMMENT ON TABLE cvterm_relationship IS 'A relationship linking two
cvterms. Each cvterm_relationship constitutes an edge in the graph
defined by the collection of cvterms and cvterm_relationships. The
meaning of the cvterm_relationship depends on the definition of the
cvterm R refered to by type_id. However, in general the definitions
are such that the statement "all SUBJs REL some OBJ" is true. The
cvterm_relationship statement is about the subject, not the
object. For example "insect wing part_of thorax".';

COMMENT ON COLUMN cvterm_relationship.type_id IS 'The nature of the
relationship between subject and object. Note that relations are also
housed in the cvterm table, typically from the OBO relationship
ontology, although other relationship types are allowed.';

COMMENT ON COLUMN cvterm_relationship.subject_id IS 'The subject of
the subj-predicate-obj sentence. The cvterm_relationship is about the
subject. In a graph, this typically corresponds to the child node.';

COMMENT ON COLUMN cvterm_relationship.object_id IS 'The object of the
subj-predicate-obj sentence. The cvterm_relationship refers to the
object. In a graph, this typically corresponds to the parent node.';

-- --------------------------------------------------------------------------
--  Indexes structure for table cvterm_relationship
-- --------------------------------------------------------------------------
CREATE INDEX  cvterm_relationship_idx1 ON cvterm_relationship USING btree(type_id ASC NULLS LAST);
CREATE INDEX  cvterm_relationship_idx2 ON cvterm_relationship USING btree(subject_id ASC NULLS LAST);
CREATE INDEX  cvterm_relationship_idx3 ON cvterm_relationship USING btree(object_id ASC NULLS LAST);

/*
|--------------------------------------------------------------------------
| CREATE cvtermpath TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvtermpath (
	cvtermpath_id serial PRIMARY KEY,
	type_id int4 REFERENCES cvterm, 
	subject_id int4 NOT NULL REFERENCES cvterm,  
	object_id int4 NOT NULL REFERENCES cvterm, 
	cv_id int4 NOT NULL REFERENCES cv,
	pathdistance int4,
	exclusive_type_id int4
);


-- --------------------------------------------------------------------------
--  Comments for table cvtermpath
-- --------------------------------------------------------------------------

COMMENT ON TABLE cvtermpath IS 'The reflexive transitive closure of
the cvterm_relationship relation.';

COMMENT ON COLUMN cvtermpath.type_id IS 'The relationship type that
this is a closure over. If null, then this is a closure over ALL
relationship types. If non-null, then this references a relationship
cvterm - note that the closure will apply to both this relationship
AND the OBO_REL:is_a (subclass) relationship.';

COMMENT ON COLUMN cvtermpath.cv_id IS 'Closures will mostly be within
one cv. If the closure of a relationship traverses a cv, then this
refers to the cv of the object_id cvterm.';

COMMENT ON COLUMN cvtermpath.pathdistance IS 'The number of steps
required to get from the subject cvterm to the object cvterm, counting
from zero (reflexive relationship).';


-- ----------------------------
--  Indexes structure for table cvtermpath
-- ----------------------------
CREATE INDEX  cvtermpath_idx1 ON cvtermpath USING btree(type_id ASC NULLS LAST);
CREATE INDEX  cvtermpath_idx2 ON cvtermpath USING btree(subject_id ASC NULLS LAST);
CREATE INDEX  cvtermpath_idx3 ON cvtermpath USING btree(object_id ASC NULLS LAST);
CREATE INDEX  cvtermpath_idx4 ON cvtermpath USING btree(cv_id ASC NULLS LAST);


/*
|--------------------------------------------------------------------------
| CREATE pub TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE pub (
	pub_id serial PRIMARY KEY,
	title text,
	volumetitle text,
	volume varchar(255),
	series_name varchar(255),
	issue varchar(255),
	pyear varchar(255),
	pages varchar(255),
	miniref varchar(255),
	uniquename text NOT NULL,
	type_id int4 NOT NULL REFERENCES cvterm,
	is_obsolete bool DEFAULT false,
	publisher varchar(255),
	pubplace varchar(255),
	title_tsvector tsvector,
	abstract text
);


-- --------------------------------------------------------------------------
--  Comments for table pub
-- --------------------------------------------------------------------------

COMMENT ON TABLE pub IS 'A documented provenance artefact - publications,
documents, personal communication.';

COMMENT ON COLUMN pub.title IS 'Descriptive general heading.';

COMMENT ON COLUMN pub.volumetitle IS 'Title of part if one of a series.';

COMMENT ON COLUMN pub.series_name IS 'Full name of (journal) series.';

COMMENT ON COLUMN pub.pages IS 'Page number range[s], e.g. 457--459, viii + 664pp, lv--lvii.';

COMMENT ON COLUMN pub.type_id IS 'The type of the publication (book, journal, poem, graffiti, etc). Uses pub cv.';


-- ----------------------------
--  Indexes structure for table pub
-- ----------------------------
CREATE INDEX  pub_idx1 ON pub USING btree(type_id ASC NULLS LAST);
CREATE INDEX  pub_title_tsvector_gin ON pub USING gin(title_tsvector) WITH (FASTUPDATE = NO);

/*
|--------------------------------------------------------------------------
| CREATE cvterm_pub TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvterm_pub (
	cvterm_pub_id serial PRIMARY KEY,
	cvterm_id int4 NOT NULL REFERENCES cvterm,
	pub_id int4 NOT NULL REFERENCES pub
);


-- --------------------------------------------------------------------------
--  Comments for table cvterm_pub
-- --------------------------------------------------------------------------





/*
|--------------------------------------------------------------------------
| CREATE models TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE models (
	model_id serial PRIMARY KEY,
	model_type_id int4 REFERENCES cvterm,
	name varchar (255) NOT NULL,
	description varchar (255) NOT NULL
);


-- --------------------------------------------------------------------------
--  Comments for table models
-- --------------------------------------------------------------------------






/*
|--------------------------------------------------------------------------
| CREATE model_parameters TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE model_parameters (
	model_parameters_id serial PRIMARY KEY,
	type_id int4 REFERENCES cvterm,
	model_id int4 REFERENCES models,
	value text NOT NULL,
	dbxref_id int4 REFERENCES dbxref
);


-- --------------------------------------------------------------------------
--  Comments for table model_parameters
-- --------------------------------------------------------------------------





/*
|--------------------------------------------------------------------------
| CREATE model_sequences TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE model_sequences (
	model_sequences_id serial PRIMARY KEY,
	model_id int4 REFERENCES models,
	dna_sequence text,
	protein_sequence text,
	source_species int4 REFERENCES dbxref,
	uniquename text,
	accession text, -- NEW FIELD FOR DNA ACCESSION FROM NCBI
	dbxref_id_protein int4 REFERENCES dbxref, -- NEW FIELD FOR PROTEIN DBXREF_ID 
	dbxref_id_dna int4 REFERENCES dbxref-- NEW FIELD FOR DNA DBXREF_ID 
);


-- --------------------------------------------------------------------------
--  Comments for table model_sequences
-- --------------------------------------------------------------------------


/*
|--------------------------------------------------------------------------
| CREATE model_sequences_dbxref TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE model_sequences_dbxref (
	model_sequences_dbxref_id serial PRIMARY KEY,
	model_sequences_id int4 REFERENCES model_sequences,
	dbxref_id int4 REFERENCES dbxref
);


-- --------------------------------------------------------------------------
--  Comments for table model_sequences_dbxref
-- --------------------------------------------------------------------------





/*
|--------------------------------------------------------------------------
| CREATE cvterm_models TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvterm_models (
	cvterm_models_id serial PRIMARY KEY,
	cvterm_id int4 REFERENCES cvterm,
	model_id int4 REFERENCES models
);


-- --------------------------------------------------------------------------
--  Comments for table cvterm_models
-- --------------------------------------------------------------------------



/*
|--------------------------------------------------------------------------
| CREATE synonym TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE synonym (
	synonym_id serial PRIMARY KEY,
	name varchar(255) NOT NULL,
	type_id int4 NOT NULL REFERENCES cvterm,
	synonym_sgml varchar(255) NOT NULL,
	name_tsvector tsvector
);


-- --------------------------------------------------------------------------
--  Comments for table synonym
-- --------------------------------------------------------------------------

COMMENT ON TABLE synonym IS 'A synonym for a feature. One feature can have multiple synonyms, and the same synonym can apply to multiple features.';

COMMENT ON COLUMN synonym.name IS 'The synonym itself. Should be human-readable machine-searchable ascii text.';

COMMENT ON COLUMN synonym.type_id IS 'Types would be symbol and fullname for now.';

COMMENT ON COLUMN synonym.synonym_sgml IS 'The fully specified synonym, with any non-ascii characters encoded in SGML.';

-- ----------------------------
--  Indexes structure for table synonym
-- ----------------------------

CREATE INDEX  name_tsvector_gin ON synonym USING gin(name_tsvector) WITH (FASTUPDATE = NO);
CREATE INDEX  synonym_idx1 ON synonym USING btree(type_id ASC NULLS LAST);
CREATE INDEX  synonym_idx2 ON synonym USING btree(lower(synonym_sgml::text) COLLATE "default" ASC NULLS LAST);


/*
|--------------------------------------------------------------------------
| CREATE cvtermsynonym TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvtermsynonym (
	cvtermsynonym_id serial PRIMARY KEY,
	cvterm_id int4 NOT NULL REFERENCES cvterm,
	synonym varchar(1024) NOT NULL,
	type_id int4 REFERENCES cvterm,
	synonym_tsvector tsvector
);

-- --------------------------------------------------------------------------
--  Comments for table cvtermsynonym
-- --------------------------------------------------------------------------


COMMENT ON TABLE cvtermsynonym IS 'A cvterm actually represents a
distinct class or concept. A concept can be refered to by different
phrases or names. In addition to the primary name (cvterm.name) there
can be a number of alternative aliases or synonyms. For example, "T
cell" as a synonym for "T lymphocyte".';

COMMENT ON COLUMN cvtermsynonym.type_id IS 'A synonym can be exact,
narrower, or broader than.';


-- ----------------------------
--  Indexes structure for table cvtermsynonym
-- ----------------------------

CREATE INDEX  cvtermsynonym_idx1 ON cvtermsynonym USING btree(cvterm_id ASC NULLS LAST);
CREATE INDEX  synonym_tsvector_gin ON cvtermsynonym USING gin(synonym_tsvector) WITH (FASTUPDATE = NO);


/*
|--------------------------------------------------------------------------
| CREATE cvtermprop TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvtermprop (
	cvtermprop_id serial PRIMARY KEY,
	cvterm_id int4 NOT NULL REFERENCES cvterm,
    type_id int4 NOT NULL REFERENCES cvterm,
	value text NOT NULL,
	rank int4 NOT NULL
);


-- --------------------------------------------------------------------------
--  Comments for table cvtermprop
-- --------------------------------------------------------------------------






/*
|--------------------------------------------------------------------------
| CREATE expression TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE expression (
	expression_id serial PRIMARY KEY,
	uniquename text NOT NULL,
	md5checksum char(32),
	description text,
	type_id int4 NOT NULL,
	description_tsvector tsvector
);

-- --------------------------------------------------------------------------
--  Comments for table expression
-- --------------------------------------------------------------------------

COMMENT ON TABLE expression IS 'The expression table is essentially a bridge table.';


/*
|--------------------------------------------------------------------------
| CREATE expression_cvterm TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE expression_cvterm (
	expression_cvterm_id serial PRIMARY KEY,
	expression_id int4 NOT NULL REFERENCES expression,
	cvterm_id int4 NOT NULL REFERENCES cvterm,
	rank int4 NOT NULL DEFAULT 0,
	cvterm_type_id int4 NOT NULL DEFAULT 1 REFERENCES cvterm
);


-- --------------------------------------------------------------------------
--  Comments for table expression_cvterm
-- --------------------------------------------------------------------------

-- ----------------------------
--  Indexes structure for table expression_cvterm
-- ----------------------------
CREATE INDEX  expression_cvterm_idx1 ON expression_cvterm USING btree(expression_id ASC NULLS LAST);
CREATE INDEX  expression_cvterm_idx2 ON expression_cvterm USING btree(cvterm_id ASC NULLS LAST);
CREATE INDEX  expression_cvterm_idx3 ON expression_cvterm USING btree(cvterm_type_id ASC NULLS LAST);




/*
|--------------------------------------------------------------------------
| CREATE cvterm_dbxref TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE cvterm_dbxref (
	cvterm_dbxref_id serial PRIMARY KEY,
	cvterm_id int4 NOT NULL REFERENCES cvterm,
	dbxref_id int4 NOT NULL REFERENCES dbxref,
	is_for_definition int4 NOT NULL DEFAULT 0
);


-- --------------------------------------------------------------------------
--  Comments for table cvterm_dbxref
-- --------------------------------------------------------------------------

COMMENT ON TABLE cvterm_dbxref IS 'In addition to the primary
identifier (cvterm.dbxref_id) a cvterm can have zero or more secondary
identifiers/dbxrefs, which may refer to records in external
databases. The exact semantics of cvterm_dbxref are not fixed. For
example: the dbxref could be a pubmed ID that is pertinent to the
cvterm, or it could be an equivalent or similar term in another
ontology. For example, GO cvterms are typically linked to InterPro
IDs, even though the nature of the relationship between them is
largely one of statistical association. The dbxref may be have data
records attached in the same database instance, or it could be a
"hanging" dbxref pointing to some external database. NOTE: If the
desired objective is to link two cvterms together, and the nature of
the relation is known and holds for all instances of the subject
cvterm then consider instead using cvterm_relationship together with a
well-defined relation.';

COMMENT ON COLUMN cvterm_dbxref.is_for_definition IS 'A
cvterm.definition should be supported by one or more references. If
this column is true, the dbxref is not for a term in an external database -
it is a dbxref for provenance information for the definition.';

-- ----------------------------
--  Indexes structure for table cvterm_dbxref
-- ----------------------------
CREATE INDEX  cvterm_dbxref_idx1 ON cvterm_dbxref USING btree(cvterm_id ASC NULLS LAST);
CREATE INDEX  cvterm_dbxref_idx2 ON cvterm_dbxref USING btree(dbxref_id ASC NULLS LAST);




/*
|--------------------------------------------------------------------------
| CREATE pub_dbxref TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE pub_dbxref (
	pub_dbxref_id serial PRIMARY KEY,
	pub_id int4 NOT NULL REFERENCES pub,
	dbxref_id int4 NOT NULL REFERENCES dbxref,
	is_current bool NOT NULL DEFAULT true
);


-- --------------------------------------------------------------------------
--  Comments for table pub_dbxref
-- --------------------------------------------------------------------------

COMMENT ON TABLE pub_dbxref IS 'Handle links to repositories,
e.g. Pubmed, Biosis, zoorec, OCLC, Medline, ISSN, coden...';

-- ----------------------------
--  Indexes structure for table pub_dbxref
-- ----------------------------
CREATE INDEX  pub_dbxref_idx1 ON pub_dbxref USING btree(pub_id ASC NULLS LAST);
CREATE INDEX  pub_dbxref_idx2 ON pub_dbxref USING btree(dbxref_id ASC NULLS LAST);


/*
|--------------------------------------------------------------------------
| CREATE downloads TABLE
| This is a table to store card downloadable json file
|--------------------------------------------------------------------------
*/

CREATE TABLE json_downloads (
	json_downloads_id serial PRIMARY KEY,
	data text,
	version text NOT NULL,
	created_at Date NOT NULL,
	filepath text NOT NULL,
	type int2,
	file_formats text
);


-- --------------------------------------------------------------------------
--  Comments for table json_downloads
-- --------------------------------------------------------------------------





/*
|--------------------------------------------------------------------------
| CREATE organism TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE organism (
	organism_id serial PRIMARY KEY,
	abbreviation varchar(255),
	genus varchar(255) NOT NULL,
	species varchar(255) NOT NULL,
	common_name varchar(255),
	comment text
);

-- --------------------------------------------------------------------------
--  Comments for table organism
-- --------------------------------------------------------------------------
COMMENT ON TABLE organism IS 'The organismal taxonomic
classification. Note that phylogenies are represented using the
phylogeny module, and taxonomies can be represented using the cvterm
module or the phylogeny module.';

COMMENT ON COLUMN organism.species IS 'A type of organism is always
uniquely identified by genus and species. When mapping from the NCBI
taxonomy names.dmp file, this column must be used where it
is present, as the common_name column is not always unique (e.g. environmental
samples). If a particular strain or subspecies is to be represented,
this is appended onto the species name. Follows standard NCBI taxonomy
pattern.';


/*
|--------------------------------------------------------------------------
| CREATE card_admin TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE card_admin (
	flag varchar(128) NOT NULL,
	value int4,
	description text NOT NULL
);

-- --------------------------------------------------------------------------
--  Comments for table card_admin
-- --------------------------------------------------------------------------



/*
|--------------------------------------------------------------------------
| CREATE pmid TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE pmid (
	pmid int4,
	priority int4,
	normalization text,
	curation int4,
	aro int4,
	pubmodule int4,
	flag int4,
	authornotify int4
);


-- --------------------------------------------------------------------------
--  Comments for table pmid
-- --------------------------------------------------------------------------


/*
|--------------------------------------------------------------------------
| CREATE tableinfo TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE tableinfo (
	tableinfo_id serial PRIMARY KEY,
	name varchar(30) NOT NULL,
	primary_key_column varchar(30),
	is_view int4 NOT NULL DEFAULT 0,
	view_on_table_id int4,
	superclass_table_id int4,
	is_updateable int4 NOT NULL DEFAULT 1,
	modification_date date NOT NULL DEFAULT now()
);

-- --------------------------------------------------------------------------
--  Comments for table tableinfo
-- --------------------------------------------------------------------------


/*
|--------------------------------------------------------------------------
| CREATE users TABLE
|--------------------------------------------------------------------------
*/


CREATE TABLE users (
	id serial PRIMARY KEY,
	name varchar(30) NOT NULL,
	email TEXT UNIQUE,
	password varchar(60) NOT NULL,
	remember_token varchar(100),
	created_at TIMESTAMP DEFAULT current_timestamp,
	updated_at TIMESTAMP DEFAULT current_timestamp
);


/*
|--------------------------------------------------------------------------
| CREATE api TABLE
|--------------------------------------------------------------------------
*/

CREATE TABLE api (
	id serial PRIMARY KEY,
	session TEXT,
	inType varchar(30),
	inputFile TEXT,
	url TEXT,
	response TEXT
);


